<?php
if (php_sapi_name() !== 'cli') {
    http_response_code(403);
    exit('Forbidden');
}
require_once __DIR__ . '/../../wp-load.php';

class StockSyncCLI {
    private string $mappingFile;
    private string $subdomain;
    private string $apiKey;
    private int $batchSize;
    private array $results = [];
    private int $found = 0;
    private int $updated = 0;
    private int $skipped = 0;

    public function __construct(string $mappingFile, string $subdomain, string $apiKey, int $batchSize = 10) {
        $this->mappingFile = $mappingFile;
        $this->subdomain = $subdomain;
        $this->apiKey = $apiKey;
        $this->batchSize = $batchSize;
    }

    public function run(): void {
        $mapping = json_decode(file_get_contents($this->mappingFile), true);
        if (!is_array($mapping)) {
            die("❌ Invalid mapping file.\n");
        }

        $total = count($mapping);
        $batches = array_chunk($mapping, $this->batchSize);
        $current = 0;

        foreach ($batches as $batch) {
            $multi = curl_multi_init();
            $handles = [];

            foreach ($batch as $entry) {
                $productId = $entry['id'];
                $barcode = $entry['barcode'];
                $url = "https://{$this->subdomain}.daftra.com/api2/products/{$productId}.json";

                $ch = curl_init($url);
                curl_setopt_array($ch, [
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_HTTPHEADER => [
                        "Accept: application/json",
                        "apikey: {$this->apiKey}"
                    ],
                    CURLOPT_TIMEOUT => 15
                ]);

                curl_multi_add_handle($multi, $ch);
                $handles[] = ['ch' => $ch, 'barcode' => $barcode];
            }

            do {
                curl_multi_exec($multi, $running);
                curl_multi_select($multi);
            } while ($running > 0);

            foreach ($handles as $item) {
                $ch = $item['ch'];
                $barcode = $item['barcode'];
                $response = curl_multi_getcontent($ch);
                curl_multi_remove_handle($multi, $ch);
                curl_close($ch);

                $data = json_decode($response, true);
                $product = $data['data']['Product'] ?? null;

                $stock = 0;
                if (!empty($product['ProductStock'])) {
                    foreach ($product['ProductStock'] as $s) {
                        $stock += floatval($s['balance'] ?? 0);
                    }
                } elseif (isset($product['stock_balance'])) {
                    $stock = floatval($product['stock_balance']);
                }

                $this->updateWooStock($barcode, $stock);
                $current++;
                $this->printProgress($current, $total);
            }

            curl_multi_close($multi);
        }

        echo PHP_EOL;
        echo "✅ Done.\n";
        echo "✔️ Found: {$this->found}, Updated: {$this->updated}, Skipped: {$this->skipped}\n";
    }

    private function updateWooStock(string $barcode, float $qty): void {
        $args = [
            'post_type' => ['product', 'product_variation'],
            'meta_query' => [
                [
                    'key' => '_global_unique_id',
                    'value' => $barcode
                ]
            ],
            'posts_per_page' => 1
        ];

        $posts = get_posts($args);
        if (!empty($posts)) {
            $this->found++;
            $pid = $posts[0]->ID;
            update_post_meta($pid, '_stock', $qty);
            update_post_meta($pid, '_stock_status', $qty > 0 ? 'instock' : 'outofstock');
            wc_delete_product_transients($pid);
            $this->updated++;
        } else {
            $this->skipped++;
        }
    }

    private function printProgress(int $current, int $total): void {
        $percent = $current / $total;
        $barLength = 30;
        $filledLength = round($barLength * $percent);
        $bar = str_repeat('#', $filledLength) . str_repeat('-', $barLength - $filledLength);
        $percentDisplay = str_pad(round($percent * 100), 3, ' ', STR_PAD_LEFT);
        echo "\r[{$bar}] {$percentDisplay}% ({$current} of {$total})";
        flush();
    }
}

// ✅ CLI-only execution
if (php_sapi_name() === 'cli') {
    $cli = new StockSyncCLI(
        __DIR__ . '/product_mapping.json',
        'infodiamondeye-egypt',
        '55d1752aacd9109225e8ba1b8b3bcc7fc0b3f642',
        10
    );
    $cli->run();
}
